-module(server).
-author('nils.muellner@gmail.com').
-export([start/0, server/9]).
-include("global_config.hrl").
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%                                                                     %%%%%
%%%%% Redistribution and use in source and binary forms, with or without  %%%%%
%%%%% modification, are permitted provided that the following conditions  %%%%%
%%%%% are met:                                                            %%%%%
%%%%%                                                                     %%%%%
%%%%% Redistributions of source code must retain the above copyright      %%%%%
%%%%% notice, this list of conditions and the following disclaimer.       %%%%%
%%%%%                                                                     %%%%%
%%%%% Redistributions in binary form must reproduce the above copyright   %%%%%
%%%%% notice, this list of conditions and the following disclaimer in     %%%%%
%%%%% the documentation and/or other materials provided with the          %%%%%
%%%%% distribution.                                                       %%%%%
%%%%%                                                                     %%%%%
%%%%% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND               %%%%%
%%%%% CONTRIBUTOR "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,          %%%%%
%%%%% INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF            %%%%%
%%%%% MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE            %%%%%
%%%%% DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE        %%%%%
%%%%% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,    %%%%%
%%%%% OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,            %%%%%
%%%%% PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,     %%%%%
%%%%% OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON         %%%%%
%%%%% ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,     %%%%%
%%%%% OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY      %%%%%
%%%%% OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE             %%%%%
%%%%% POSSIBILITY OF SUCH DAMAGE.                                         %%%%%
%%%%%                                                                     %%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
start() ->
	io:format("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%                                                                     %%%%%
%%%%%                                SiSSDA                               %%%%%
%%%%%                                                                     %%%%%
%%%%%                                 v ~p                             %%%%%
%%%%%                                                                     %%%%%
%%%%%                                                                     %%%%%
%%%%%                  Welcome to the Simulator for                       %%%%%
%%%%%             Self-Stabilizing Distributed Algorithms                 %%%%%
%%%%%                                                                     %%%%%
%%%%%                                SERVER                               %%%%%
%%%%%                                                                     %%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%~n",[?version]),
	file:delete(?log_file),
	Phase = alg,
	Algorithm = unset,
	Client_Data = unset,
	Server_Data = unset,
	Exec_Sem = ?exec_sem,
	GPP = general_purpose_register,
	Counter = {0, 0},
	{ok, File} = file:open(?log_file, [write, append]),
	register(server, spawn(server, server, [Phase, Algorithm, Client_Data, Server_Data, Exec_Sem, GPP, Counter, ?accuracy_field, File])).

server(Phase, Algorithm, Client_Data, Server_Data, Exec_Sem, GPP, Counter, Acc_Field, File)->
	if
		Phase == alg ->
			io:format("%%%%% INITIALIZATION-PHASE 1: CHOOSE ALGORITHM                            %%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%~n"),
			io:format(""),
			Available_Algorithms = init_get_algorithms(?tops,[],[]),
			io:format("%%%%% The following algorithms are available:                             %%%%%~n"),
			init_print_algorithms(Available_Algorithms, 0),
			io:format("%%%%% Please enter the appropriate number [n.]"),
			{_, Chosen_Alg} = io:read(standard_io, '>'),
			Number_Algs = length(Available_Algorithms),
			if
				is_integer(Chosen_Alg), Chosen_Alg > 0, Chosen_Alg < (Number_Algs + 1) ->
					Accepted = true;
				true ->
					Accepted = false
			end,
			if
				Accepted == true ->
					io:format("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%~n"),
					New_Algorithm = lists:nth(Chosen_Alg, Available_Algorithms),
					New_Phase = topology;
				true ->
					New_Algorithm = Algorithm,
					New_Phase = Phase
			end,
			server(New_Phase, New_Algorithm, Client_Data, Server_Data, Exec_Sem, GPP, Counter, Acc_Field, File);
		Phase == topology ->
			io:format("%%%%% INITIALIZATION-PHASE 2: CHOOSE TOPOLOGY                             %%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%~n"),
			Available_Tops = topology_get_tops(?tops, Algorithm, []),
			io:format("%%%%% The following topologies are available:                             %%%%%~n"),
			if
				?verbose == true; ?verbose == topology ->
					topology_show_top_long(Available_Tops, 0);
				true ->
					topology_show_top_short(Available_Tops, 0)
			end,
			io:format("%%%%% Please enter the appropriate number [n.]"),
			{_, Chosen_Top} = io:read(standard_io, '>'),
			Number_Tops = length(Available_Tops),
			if
				is_integer(Chosen_Top), Chosen_Top > 0, Chosen_Top < (Number_Tops + 1) ->
					Accepted = true;
				true ->
					Accepted = false
			end,
			if
				Accepted == true ->
					io:format("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%~n"),
					New_Client_Data = lists:nth(Chosen_Top, Available_Tops),
					New_Phase = augmenting;
				true ->
					New_Client_Data = Client_Data,
					New_Phase = Phase
			end,
			server(New_Phase, Algorithm, New_Client_Data, Server_Data, Exec_Sem, GPP, Counter, Acc_Field, File);
		Phase == augmenting ->
			[_,_|Top] = Client_Data,
			{Number_of_Clients, Temp_Client_Data, New_Server_Data} = matrix_init:start(Algorithm, Top),
			if
				Algorithm == dfs ->
					New_Client_Data = Temp_Client_Data;
				true ->
					New_Client_Data = switch_arrows(Temp_Client_Data)
			end,
			io:format("%%%%%% PLEASE CONNECT ~p CLIENTS...~n", [Number_of_Clients]),
			New_Phase = connect_clients,
			server(New_Phase, Algorithm, New_Client_Data, New_Server_Data, Exec_Sem, GPP, Counter, Acc_Field, File);
		Phase == connect_clients ->
			receive
				{connect, PID} ->
					{Name, Space_Left, New_Client_Data} = connect_client(PID, Client_Data),
					io:format("%%%%% Client ~p is now identified as process ~p~n", [Name, PID]),
					PID ! {success, Name, Algorithm}
			end,
			if
				Space_Left > 0 ->
					io:format("%%%%% Please connect ~p more clients.~n", [Space_Left]),
					New_Phase = Phase;
				true ->
					New_Phase = fault_injector,
					io:format("%%%%% SUFFICIENT NUMBER OF CLIENTS CONNECTED                              %%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%% PLEASE START THE FAULT-INJECTOR NOW                                 %%%%%~n"),
					receive
						{fault_injector, connect} ->
							io:format("%%%%% FAULT-INJECTOR CONNECTED SUCCESSFULLY                               %%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%
%%%%%      SYSTEM IS NOW RUNNING. PLEASE WAIT...
%%%%%
%%%%%~n"),
						?fault_injector ! ack
					end
			end,
			server(New_Phase, Algorithm, New_Client_Data, Server_Data, Exec_Sem, GPP, Counter, Acc_Field, File);
		Phase == fault_injector ->
			?fault_injector ! {Algorithm, Client_Data},
			receive
				New_Client_Data ->
					ok
			end,
			New_Phase = phase_6,
			server(New_Phase, Algorithm, New_Client_Data, Server_Data, Exec_Sem, GPP, Counter, Acc_Field, File);
		Phase == phase_0 ->
			receive
				{ready, Name}->
					New_GPP = lists:append(GPP, [Name])
			end,
			Length1 = length(Client_Data),
			Length2 = length(New_GPP),
			if
				Length1 == Length2 ->
					New_Phase = phase_1;
				true ->
					New_Phase = Phase
			end,
			server(New_Phase, Algorithm, Client_Data, Server_Data, Exec_Sem, New_GPP, Counter, Acc_Field, File);
		Phase == phase_1 ->
			?fault_injector ! {Algorithm, Client_Data, Exec_Sem},
			receive
				{Exec_List, New_Client_Data} ->
					New_Phase = phase_2,
					New_Exec_List = phase_1_build_list(Exec_List, New_Client_Data, []);
				kill ->
					New_Client_Data = Client_Data,
					New_Phase = sudden_death,
					New_Exec_List = unset;
				pause ->
					receive
						{Exec_List, New_Client_Data} ->
							New_Phase = phase_2,
							New_Exec_List = phase_1_build_list(Exec_List, New_Client_Data, [])
					end,
					receive
						pause ->
							ok
					end
			end,
			server(New_Phase, Algorithm, New_Client_Data, Server_Data, Exec_Sem, New_Exec_List, Counter, Acc_Field, File);
		Phase == phase_2 ->
			[First| Rest] = GPP,
			{PID, Paket} = First,
			PID ! Paket,
			Length = length(Rest),
			if
				Length == 0 ->
					New_Phase = phase_4;
				true ->
					New_Phase = Phase
			end,
			server(New_Phase, Algorithm, Client_Data, Server_Data, Exec_Sem, Rest, Counter, Acc_Field, File);
		Phase == phase_4 ->
			receive
				{Name, New_Current_Value} ->
					New_GPP = lists:append(GPP, [Name]),
					New_Client_Data = phase_4_update(Client_Data, Name, New_Current_Value)
			end,
			Length1 = length(New_Client_Data),
			Length2 = length(New_GPP),
			if
				Length1 == Length2 ->
					New_Phase = phase_5;
				true ->
					New_Phase = Phase
			end,
			server(New_Phase, Algorithm, New_Client_Data, Server_Data, Exec_Sem, New_GPP, Counter, Acc_Field, File);
		Phase == phase_5 ->
			New_Phase = phase_6,
			{New_Algorithm, New_Client_Data, New_Server_Data, New_Exec_Sem} = fault_env:start(Algorithm, Client_Data, Server_Data, Exec_Sem),
			server(New_Phase, New_Algorithm, New_Client_Data, New_Server_Data, New_Exec_Sem, GPP, Counter, Acc_Field, File);
		Phase == phase_6 ->
			?fault_injector ! {update, Client_Data},
			receive
				{updated, New_Server_Data} ->
					ok
			end,
			if
				?verbose == true ->
					io:format("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%% RECORDING VALUES:                                                   %%%%%
~p
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%~n", [New_Server_Data]);
				true ->
					ok
			end,
			New_Counter = phase_6_counter(Client_Data, Counter, Algorithm),
			{Accuracy_Reached, New_Acc_Field} = phase_6_accuracy(Acc_Field, New_Counter),
			{NCL, NCI} = New_Counter,
			Steps = NCL + NCI,
			if
				Steps rem ?accuracy_min_run_length == 0 ->
					Elem = lists:last(New_Acc_Field),
					io:format("%%%%% STEP: ~p     LEGAL: ~p~n", [Steps, Elem]);
				true ->
					ok
			end,
			if
				Accuracy_Reached == true ->
					New_Phase = kill;
				true ->
					phase_6_wake_clients(Client_Data, Client_Data, Algorithm),
					New_Phase = phase_0
			end,
			server(New_Phase, Algorithm, Client_Data, New_Server_Data, Exec_Sem, [], New_Counter, New_Acc_Field, File);
		Phase == kill ->
			file:close(File),
			[First| Rest] = Client_Data,
			{_, _, PID, _, _, _, _} = First,
			PID ! kill,
			receive
				{killed, Name} ->
					io:format("%%%%% Successfully terminated node ~p~n",[Name])
			end,
			Length = length(Rest),
			if
				Length > 0 ->
					New_Phase = Phase;
				true ->
					New_Phase = going_down
			end,
			server(New_Phase, Algorithm, Rest, Server_Data, Exec_Sem, [], Counter, Acc_Field, File);
		Phase == going_down ->
			?fault_injector ! kill,
			receive
				{killed, fault_injector} ->
					("%%%%% Successfully terminated fault injector                              %%%%%~n")
			end,
			{Legal, Illegal} = Counter,
			Sum = Legal + Illegal,
			Result = Legal / Sum,
			io:format("%%%%% RESULT: ~p
%%%%% STEPS: ~p
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%~n", [Result, Sum]),
			io:format("%%%%% ALL CLIENTS/FI KILLED. SERVER GOING DOWN, TOO...                    %%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%~n"),
			exit(normal);
		Phase == sudden_death ->
			io:format("%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%% RECEIVED EXTRAORDINARY SUDDEN DEATH SIGNAL                          %%%%%
%%%%% WILL GO DOWN UNPLANNED...                                           %%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%~n"),
			New_Phase = kill,
			server(New_Phase, Algorithm, Client_Data, Server_Data, Exec_Sem, [], Counter, Acc_Field, File)
	end.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%         NILS' LITTLE HELPER FUNCTIONS         %%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
connect_client(PID, Client_Data)->
	Next_Free_Slot = connect_client_get_slot(Client_Data),
	Temp_Client_Data = lists:delete(Next_Free_Slot, Client_Data),
	{Name, FP, _, Neighs, Cur, Stab, Wait} = Next_Free_Slot,
	New_Elem = {Name, FP, PID, Neighs, Cur, Stab, Wait},
	New_Client_Data = lists:sort(lists:append(Temp_Client_Data, [New_Elem])),
	Space_Left = connect_client_space_left(New_Client_Data, 0),
	{Name, Space_Left, New_Client_Data}.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
connect_client_get_slot(Server_Data)->
	[First| Rest] = Server_Data,
	{_, _, PID, _, _, _, _} = First,
	if
		PID == pid ->
			First;
		true ->
			connect_client_get_slot(Rest)
	end.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
connect_client_space_left([], Counter)->
	Counter;
connect_client_space_left(New_Server_Data, Counter)->
	[First| Rest] = New_Server_Data,
	{_, _, PID, _, _, _, _} = First,
	if
		PID == pid ->
			New_Counter = Counter + 1;
		true ->
			New_Counter = Counter
	end,
	connect_client_space_left(Rest, New_Counter).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
get_elem(Name, Data)->
	[First| Rest] = Data,
	{Node, _, _, _, _, _, _} = First,
	if
		Node == Name ->
			First;
		true ->
			get_elem(Name, Rest)
	end.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
init_get_algorithms([], Algorithms,[])->
	Sorted_Algorithms = lists:sort(Algorithms),
	Sorted_Algorithms;
init_get_algorithms(Tops, Algorithms, [])->
    [First| Rest] = Tops,
    [_, Current| _] = First,
    [First_Cur| Rest_Cur] = Current,
    Member_Check = lists:member(First_Cur, Algorithms),
    if
    	Member_Check == true ->
            New_Algorithms = Algorithms;
		true ->
            New_Algorithms = lists:append([First_Cur], Algorithms)
	end,
	init_get_algorithms(Rest, New_Algorithms, Rest_Cur);
init_get_algorithms(Tops, Algorithms, Current)->
	[First_Cur| Rest_Cur] = Current,
	Member_Check = lists:member(First_Cur, Algorithms),
	if
		Member_Check == true ->
			New_Algorithms = Algorithms;
		true ->
			New_Algorithms = lists:append([First_Cur], Algorithms)
	end,
	init_get_algorithms(Tops, New_Algorithms, Rest_Cur).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
init_print_algorithms([], _)->
	ok;
init_print_algorithms(Available_Algorithms, Counter)->
	New_Counter = Counter + 1,
	[First| Rest] = Available_Algorithms,
	io:format("%%%%%    [~p]~p~n",[New_Counter, First]),
	init_print_algorithms(Rest, New_Counter).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
phase_1_build_list(Exec_List, [], Out)->
	New_Out = lists:append(Out, Exec_List),
	New_Out;
phase_1_build_list(Exec_List, Client_Data, Out)->
	[First| Rest] = Client_Data,
	{_, _, PID, _, _, _, _} = First,
	Bool = phase_1_inlist(PID,Exec_List),
	if
		Bool == true ->
			phase_1_build_list(Exec_List, Rest, Out);
		true ->
			New_Elem =  {PID, idle},
			New_Out = lists:append(Out, [New_Elem]),
			phase_1_build_list(Exec_List, Rest, New_Out)
	end.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
phase_1_inlist(_, []) ->
	false;
phase_1_inlist(PID,Exec_List) ->
	[First| Rest] = Exec_List,
	{Client, _} = First,
	if
		PID == Client ->
			true;
		true ->
			phase_1_inlist(PID, Rest)
	end.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
phase_4_update(Client_Data, Name, New_Current_Value)->
	Elem = get_elem(Name, Client_Data),
	Temp_Client_Data = lists:delete(Elem, Client_Data),
	{Name, NF, PID, Neighs, _, Stab, Wait} = Elem,
	New_Elem = {Name, NF, PID, Neighs, New_Current_Value, Stab, Wait},
	New_Client_Data = lists:append(Temp_Client_Data, [New_Elem]),
	Return = lists:keysort(1, New_Client_Data),
	Return.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
phase_6_accuracy(Acc_Field, New_Counter) ->
	Temp_Acc_Field = lists:delete(lists:last(Acc_Field), Acc_Field),
	{Stable, Unstable} = New_Counter,
	Steps = Stable + Unstable,
	Legal = Stable / Steps,
	New_Acc_Field = lists:append([Legal], Temp_Acc_Field),
	Min = lists:min(New_Acc_Field),
	Max = lists:max(New_Acc_Field),
	Diff = Max - Min,
	New_Diff = Diff * 1000000,
	Bool = lists:member(0, New_Acc_Field),
	if
		Bool == true; New_Diff > ?accuracy; Steps < ?accuracy_min_run_length ->
			{false, New_Acc_Field};
		true ->
			{true, New_Acc_Field}
	end.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
phase_6_bfs_dfs_le_check([])->
	true;
phase_6_bfs_dfs_le_check(Client_Data)->
	[First| Rest] = Client_Data,
	{_, _, _, _, Cur, Stab, _} = First,
	if
		Cur /= Stab ->
			false;
		true ->
			phase_6_bfs_dfs_le_check(Rest)
	end.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
phase_6_counter(Client_Data, Counter, Algorithm)->
	if
		Algorithm == bfs; Algorithm == dfs; Algorithm == le->
			Stability = phase_6_bfs_dfs_le_check(Client_Data);
		Algorithm == mutex ->
			Stability = phase_6_mutex_check(Client_Data)
	end,
	{Legal, Illegal} = Counter,
	if
		Stability == true ->
			New_Legal = Legal + 1,
			New_Illegal = Illegal;
		true ->
			New_Legal = Legal,
			New_Illegal = Illegal + 1
	end,
	New_Counter = {New_Legal, New_Illegal},
	New_Counter.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	
phase_6_mutex_check(Client_Data)->
	Values = phase_6_mutex_check_sub(Client_Data, []),
	Return = phase_6_mutex_check_sub2(Values, unused),
	Return.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	
phase_6_mutex_check_sub([], Values)->
	Values;
phase_6_mutex_check_sub(Client_Data, Values)->
	[First| Rest] = Client_Data,
	{_, _, _, _, Cur, _, _} = First,
	New_Values = lists:append(Values, [Cur]),
	phase_6_mutex_check_sub(Rest, New_Values).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
phase_6_mutex_check_sub2(_, false)->
	false;
phase_6_mutex_check_sub2([], _)->
	true;
phase_6_mutex_check_sub2(Data, Token)->
	[First, Second| Rest] = Data,
	Diff = First - Second,
	if
		Token == unused, Diff == 0 ->
			New_Token = unused;
		Token == unused, Diff /= 0 ->
			New_Token = used;
		Token == used, Diff == 0 ->
			New_Token = unused;
		Token == used, Diff /= 0 ->
			New_Token = false
	end,
	Length = length(Rest),
	if
		Length == 0 ->
			New_Data = [];
		true ->
			[_| New_Data] = Data
	end,
	phase_6_mutex_check_sub2(New_Data, New_Token).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	
phase_6_wake_clients(_, [], _)->
	ok;
phase_6_wake_clients(Client_Data, List, Algorithm)->
	[First| Rest] = List,
	{_, _, PID, _, _, _, _} = First,
	PID ! {update, Client_Data, Algorithm},
	phase_6_wake_clients(Client_Data, Rest, Algorithm).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
switch_arrows(Client_Data) ->
	{Small, Big} = switch_arrows_1(Client_Data, [], []),
	Temp = switch_arrrows_1_1(Small, Big),
	New_Client_Data = lists:keysort(1, Temp),
	New_Client_Data.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
switch_arrows_1([], Small, Big)->
	{Small, Big};
switch_arrows_1(Client_Data, Small, Big)->
	[First| Rest] = Client_Data,
	{Name, Fault, Pid, Neighs, Cur, Stab, Step} = First,
	New_Small_Elem = {Name, Neighs},
	New_Big_Elem = {Name, Fault, Pid, [], Cur, Stab, Step},
	New_Small = lists:append(Small, [New_Small_Elem]),
	New_Big = lists:append(Big, [New_Big_Elem]),
	switch_arrows_1(Rest, New_Small, New_Big).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
switch_arrrows_1_1([], Big)->
	Big;
switch_arrrows_1_1(Small, Big)->
	[First| Rest] = Small,
	{Name, Neighs} = First,
	New_Big = switch_arrrows_1_2(Name, Neighs, Big),
	switch_arrrows_1_1(Rest, New_Big).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
switch_arrrows_1_2(_, [], Big)->
	Big;
switch_arrrows_1_2(Name, Neighs, Big)->
	[First| Rest] =  Neighs,
	{Node, Fault} = First,
	Elem = get_elem(Node, Big),
	Temp_Big = lists:delete(Elem, Big),
	{_ , Fault, Pid, Old_Neighs, Cur, Stab, Step} = Elem,
	New_Neigh = {Name, Fault},
	New_Neighs = lists:append(Old_Neighs, [New_Neigh]),
	New_Elem = {Node, Fault, Pid, New_Neighs, Cur, Stab, Step},
	New_Big = lists:append(Temp_Big, [New_Elem]),
	switch_arrrows_1_2(Name, Rest, New_Big).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
topology_get_tops([], _, Available)->
	New_Available = lists:sort(Available),
	New_Available;
topology_get_tops(Topologies, Algorithm, Available)->
	[First| Rest] = Topologies,
	[_, Algs| _] = First,
	Eligible = lists:member(Algorithm, Algs),
	if
		Eligible == true ->
			New_Available = lists:append([First], Available);
		true ->
			New_Available = Available
	end,
	topology_get_tops(Rest, Algorithm, New_Available).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
topology_show_top_long([], _)->
	ok;
topology_show_top_long(Available_Tops, Counter)->
	New_Counter = Counter + 1,
	[First| Rest] = Available_Tops,
	[Name, _| Top_Rest] = First,
	io:format("%%%%%    [~p]~p~n",[New_Counter, Name]),
	topology_show_top_long_sub(Top_Rest),
	topology_show_top_long(Rest, New_Counter).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
topology_show_top_long_sub([])->
	ok;
topology_show_top_long_sub(Top_Rest)->
	[First| Rest] = Top_Rest,
    io:format("%%%%%        ~p~n",[First]),
	topology_show_top_long_sub(Rest).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
topology_show_top_short([], _)->
	ok;
topology_show_top_short(Available_Tops, Counter)->
	New_Counter = Counter + 1,
	[First| Rest] = Available_Tops,
	[Name, _| _] = First,
	io:format("%%%%%    [~p]~p~n",[New_Counter, Name]),
	topology_show_top_short(Rest, New_Counter).
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
